const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const { Pool } = require('pg');
const http = require('http');
const { Server } = require('socket.io');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const Joi = require('joi');
require('dotenv').config();

const app = express();
const server = http.createServer(app);
const io = new Server(server, {
  cors: {
    origin: process.env.CORS_ORIGIN || "*",
    methods: ["GET", "POST"]
  }
});

// Database connection
const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

// Middleware
app.use(helmet());
app.use(cors());
app.use(express.json());

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
});
app.use('/api/', limiter);

// Socket.io for real-time updates
io.on('connection', (socket) => {
  console.log('User connected:', socket.id);
  
  socket.on('disconnect', () => {
    console.log('User disconnected:', socket.id);
  });
});

// Broadcast function for real-time updates
const broadcastUpdate = (event, data, userId) => {
  // In a real app, you'd want to emit only to the specific user's room
  // For now, we'll emit to all but client side should filter or we add rooms later
  io.emit(event, data);
};

// Initialize database tables
const initDatabase = async () => {
  try {
    await pool.query(`
      CREATE TABLE IF NOT EXISTS users (
        id SERIAL PRIMARY KEY,
        username VARCHAR(50) UNIQUE NOT NULL,
        email VARCHAR(100) UNIQUE NOT NULL,
        password_hash VARCHAR(255) NOT NULL,
        telegram_chat_id VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    await pool.query(`
      CREATE TABLE IF NOT EXISTS tasks (
        id SERIAL PRIMARY KEY,
        user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        message TEXT NOT NULL,
        pomodoros INTEGER DEFAULT 1,
        duration VARCHAR(20) DEFAULT '25m',
        tags TEXT DEFAULT '',
        status VARCHAR(20) DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    await pool.query(`
      CREATE TABLE IF NOT EXISTS pomodoro_sessions (
        id SERIAL PRIMARY KEY,
        task_id INTEGER REFERENCES tasks(id) ON DELETE CASCADE,
        start_time TIMESTAMP NOT NULL,
        end_time TIMESTAMP,
        duration INTEGER NOT NULL,
        completed BOOLEAN DEFAULT FALSE,
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    console.log('Database tables initialized successfully');
  } catch (error) {
    console.error('Error initializing database:', error);
  }
};

// Validation schemas
const registerSchema = Joi.object({
  username: Joi.string().alphanum().min(3).max(30).required(),
  email: Joi.string().email().required(),
  password: Joi.string().min(6).required()
});

const loginSchema = Joi.object({
  email: Joi.string().email().required(),
  password: Joi.string().required()
});

const taskValidation = Joi.object({
  message: Joi.string().min(1).max(500).required(),
  pomodoros: Joi.number().integer().min(1).max(100),
  duration: Joi.string().pattern(/^\d+[mh]$/),
  tags: Joi.string().allow('').max(200),
  status: Joi.string().valid('active', 'completed', 'archived')
});

// Auth Middleware
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) return res.status(401).json({ error: 'Access denied' });

  jwt.verify(token, process.env.JWT_SECRET || 'your_jwt_secret', (err, user) => {
    if (err) return res.status(403).json({ error: 'Invalid token' });
    req.user = user;
    next();
  });
};

// Routes

app.get('/api/health', (req, res) => {
  res.json({ status: 'OK', timestamp: new Date().toISOString() });
});

// Auth Routes
app.post('/api/auth/register', async (req, res) => {
  try {
    const { error } = registerSchema.validate(req.body);
    if (error) return res.status(400).json({ error: error.details[0].message });

    const { username, email, password } = req.body;

    // Check if user exists
    const userCheck = await pool.query('SELECT * FROM users WHERE email = $1 OR username = $2', [email, username]);
    if (userCheck.rows.length > 0) {
      return res.status(400).json({ error: 'User already exists' });
    }

    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(password, salt);

    // Create user
    const result = await pool.query(
      'INSERT INTO users (username, email, password_hash) VALUES ($1, $2, $3) RETURNING id, username, email, created_at',
      [username, email, hashedPassword]
    );

    const user = result.rows[0];
    const token = jwt.sign({ id: user.id, username: user.username }, process.env.JWT_SECRET || 'your_jwt_secret', { expiresIn: '1y' });

    res.status(201).json({ user, token });
  } catch (error) {
    console.error('Error registering user:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.post('/api/auth/login', async (req, res) => {
  try {
    const { error } = loginSchema.validate(req.body);
    if (error) return res.status(400).json({ error: error.details[0].message });

    const { email, password } = req.body;

    // Find user
    const result = await pool.query('SELECT * FROM users WHERE email = $1', [email]);
    if (result.rows.length === 0) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    const user = result.rows[0];

    // Check password
    const validPassword = await bcrypt.compare(password, user.password_hash);
    if (!validPassword) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    const token = jwt.sign({ id: user.id, username: user.username }, process.env.JWT_SECRET || 'your_jwt_secret', { expiresIn: '1y' });

    res.json({ 
      user: { id: user.id, username: user.username, email: user.email }, 
      token 
    });
  } catch (error) {
    console.error('Error logging in:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Protected Task Routes
app.get('/api/tasks', authenticateToken, async (req, res) => {
  try {
    const result = await pool.query(
      'SELECT * FROM tasks WHERE user_id = $1 ORDER BY created_at DESC',
      [req.user.id]
    );
    res.json(result.rows);
  } catch (error) {
    console.error('Error fetching tasks:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.post('/api/tasks', authenticateToken, async (req, res) => {
  try {
    const { error } = taskValidation.validate(req.body);
    if (error) return res.status(400).json({ error: error.details[0].message });

    const { message, pomodoros = 1, duration = '25m', tags = '' } = req.body;
    
    const result = await pool.query(
      'INSERT INTO tasks (user_id, message, pomodoros, duration, tags) VALUES ($1, $2, $3, $4, $5) RETURNING *',
      [req.user.id, message, pomodoros, duration, tags]
    );

    const newTask = result.rows[0];
    broadcastUpdate('task_created', newTask, req.user.id);
    
    res.status(201).json(newTask);
  } catch (error) {
    console.error('Error creating task:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.put('/api/tasks/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { message, pomodoros, duration, tags, status } = req.body;
    
    const result = await pool.query(
      'UPDATE tasks SET message = $1, pomodoros = $2, duration = $3, tags = $4, status = $5, updated_at = CURRENT_TIMESTAMP WHERE id = $6 AND user_id = $7 RETURNING *',
      [message, pomodoros, duration, tags, status, id, req.user.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Task not found or unauthorized' });
    }

    const updatedTask = result.rows[0];
    broadcastUpdate('task_updated', updatedTask, req.user.id);
    
    res.json(updatedTask);
  } catch (error) {
    console.error('Error updating task:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.delete('/api/tasks/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    const result = await pool.query(
      'DELETE FROM tasks WHERE id = $1 AND user_id = $2 RETURNING *',
      [id, req.user.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Task not found or unauthorized' });
    }

    broadcastUpdate('task_deleted', { id }, req.user.id);
    res.json({ message: 'Task deleted successfully' });
  } catch (error) {
    console.error('Error deleting task:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Session Routes (Protected)
app.get('/api/tasks/:id/sessions', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    // Check task ownership first
    const taskCheck = await pool.query('SELECT id FROM tasks WHERE id = $1 AND user_id = $2', [id, req.user.id]);
    if (taskCheck.rows.length === 0) {
      return res.status(404).json({ error: 'Task not found or unauthorized' });
    }

    const result = await pool.query(
      'SELECT * FROM pomodoro_sessions WHERE task_id = $1 ORDER BY start_time DESC',
      [id]
    );
    res.json(result.rows);
  } catch (error) {
    console.error('Error fetching sessions:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.post('/api/tasks/:id/sessions', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { start_time, duration } = req.body;
    
    // Check task ownership
    const taskCheck = await pool.query('SELECT id FROM tasks WHERE id = $1 AND user_id = $2', [id, req.user.id]);
    if (taskCheck.rows.length === 0) {
      return res.status(404).json({ error: 'Task not found or unauthorized' });
    }

    const result = await pool.query(
      'INSERT INTO pomodoro_sessions (task_id, start_time, duration) VALUES ($1, $2, $3) RETURNING *',
      [id, start_time, duration]
    );

    const newSession = result.rows[0];
    broadcastUpdate('session_started', newSession, req.user.id);
    
    res.status(201).json(newSession);
  } catch (error) {
    console.error('Error creating session:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.put('/api/sessions/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { end_time, completed, notes } = req.body;
    
    // Join with tasks to verify ownership
    const result = await pool.query(
      `UPDATE pomodoro_sessions ps
       SET end_time = $1, completed = $2, notes = $3 
       FROM tasks t
       WHERE ps.task_id = t.id AND ps.id = $4 AND t.user_id = $5
       RETURNING ps.*`,
      [end_time, completed, notes, id, req.user.id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Session not found or unauthorized' });
    }

    const updatedSession = result.rows[0];
    broadcastUpdate('session_completed', updatedSession, req.user.id);
    
    res.json(updatedSession);
  } catch (error) {
    console.error('Error updating session:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.get('/api/stats', authenticateToken, async (req, res) => {
  try {
    const totalTasks = await pool.query('SELECT COUNT(*) FROM tasks WHERE user_id = $1', [req.user.id]);
    const completedSessions = await pool.query(`
      SELECT COUNT(*) FROM pomodoro_sessions ps
      JOIN tasks t ON ps.task_id = t.id
      WHERE t.user_id = $1 AND ps.completed = TRUE
    `, [req.user.id]);
    
    const totalPomodoroTime = await pool.query(`
      SELECT SUM(ps.duration) FROM pomodoro_sessions ps
      JOIN tasks t ON ps.task_id = t.id
      WHERE t.user_id = $1 AND ps.completed = TRUE
    `, [req.user.id]);
    
    const tasksByStatus = await pool.query('SELECT status, COUNT(*) FROM tasks WHERE user_id = $1 GROUP BY status', [req.user.id]);

    res.json({
      totalTasks: totalTasks.rows[0].count,
      completedSessions: completedSessions.rows[0].count,
      totalPomodoroTime: totalPomodoroTime.rows[0].sum || 0,
      tasksByStatus: tasksByStatus.rows
    });
  } catch (error) {
    console.error('Error fetching stats:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

const PORT = process.env.PORT || 3000;

// Start server
const startServer = async () => {
  await initDatabase();
  
  server.listen(PORT, () => {
    console.log(`Pomo API server running on port ${PORT}`);
    console.log(`Health check: http://localhost:${PORT}/api/health`);
  });
};

startServer().catch(console.error);

module.exports = { app, pool };
