# Zoom-Aware Responsive Design Guide

This guide explains how to handle browser zoom levels (like 150% zoom on 1920x1080 displays) in your website.

## The Problem

When users zoom their browser (e.g., 150% zoom on a 1920x1080 display), the effective viewport becomes smaller (1280x720), but CSS media queries still use the original viewport size. This can cause layout issues.

## Solutions Implemented

### 1. Fluid Typography

Use fluid font sizes that scale smoothly with viewport and zoom:

```tsx
// Instead of fixed sizes
<h1 className="text-5xl md:text-7xl">Title</h1>

// Use fluid sizes
<h1 className="text-fluid-5xl md:text-fluid-7xl">Title</h1>
// Or with Tailwind utilities
<h1 className="text-[clamp(3rem,2.4rem+3vw,4rem)]">Title</h1>
```

### 2. Fluid Spacing

Use fluid spacing for padding and margins:

```tsx
// Instead of fixed padding
<div className="px-4 py-8">

// Use fluid spacing
<div className="px-fluid-md py-fluid-lg">
// Or with Tailwind utilities
<div className="p-[clamp(1rem,0.8rem+1vw,1.5rem)]">
```

### 3. Zoom-Aware Breakpoints

Use the new zoom-aware breakpoints in Tailwind:

```tsx
// Standard breakpoints (may not work well at 150% zoom)
<div className="grid md:grid-cols-2 lg:grid-cols-3">

// Zoom-aware breakpoints
<div className="grid zoom-md:grid-cols-2 zoom-lg:grid-cols-3">
```

### 4. Container Queries

Use container queries for zoom-aware layouts:

```tsx
import { ZoomAwareContainer } from '@/components/ui/ZoomAwareContainer';

<ZoomAwareContainer className="max-w-7xl mx-auto">
  <div className="container-sm:grid-cols-2 container-md:grid-cols-3">
    {/* Content */}
  </div>
</ZoomAwareContainer>
```

### 5. Zoom Detection Hook

Use the hook to conditionally apply classes based on zoom level:

```tsx
import { useZoomLevel, useZoomAwareBreakpoints } from '@/hooks/useZoomLevel';

function MyComponent() {
  const zoomLevel = useZoomLevel();
  const { isMobile, isTablet, effectiveWidth } = useZoomAwareBreakpoints();
  
  return (
    <div className={zoomLevel > 1.25 ? 'text-sm' : 'text-base'}>
      {/* Content */}
    </div>
  );
}
```

## Best Practices

1. **Prefer Fluid Typography**: Always use `text-fluid-*` classes or `clamp()` for headings and important text.

2. **Use Relative Units**: Prefer `rem`, `em`, and viewport units (`vw`, `vh`) over fixed `px` values.

3. **Test at Different Zoom Levels**: Test your site at 100%, 125%, 150%, and 175% zoom.

4. **Container Queries for Components**: Use container queries for component-level responsiveness instead of viewport queries when possible.

5. **Flexible Grids**: Use `minmax()` in CSS Grid for flexible layouts:
   ```css
   grid-template-columns: repeat(auto-fit, minmax(min(300px, 100%), 1fr));
   ```

## Common Patterns

### Responsive Grid with Zoom Support

```tsx
<div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-fluid-md">
  {/* Cards */}
</div>
```

### Responsive Typography

```tsx
<h1 className="text-fluid-4xl md:text-fluid-5xl lg:text-fluid-6xl font-bold">
  Heading
</h1>
```

### Responsive Spacing

```tsx
<section className="px-fluid-md py-fluid-lg md:px-fluid-lg md:py-fluid-xl">
  {/* Content */}
</section>
```

## Migration Checklist

When updating existing components:

- [ ] Replace fixed font sizes with fluid sizes (`text-fluid-*`)
- [ ] Replace fixed padding/margins with fluid spacing (`px-fluid-*`, `py-fluid-*`)
- [ ] Add zoom-aware breakpoints where needed (`zoom-md:`, `zoom-lg:`)
- [ ] Wrap complex layouts in `ZoomAwareContainer`
- [ ] Test at 100%, 125%, 150%, and 175% zoom levels

## Browser Support

- Container Queries: Modern browsers (Chrome 105+, Firefox 110+, Safari 16+)
- CSS `clamp()`: All modern browsers
- Zoom detection: All browsers (uses standard APIs)

