'use client';

import React, { useState, useEffect } from 'react';
import { Header } from './Header';
import { Footer } from './Footer';
import { SecondBrainBanner } from './SecondBrainBanner';
import { SmoothScroll } from './SmoothScroll';
import GradientBackground from './animations/GradientBackground';
import LivingBackground from './animations/LivingBackground';
import GradientBackgroundBlackMagic from './animations/GradientBackground-blackMagic';
import { BackgroundProvider } from '../context/BackgroundContext';
import type { BackgroundMode } from '../types/theme';
import { useZoomLevel } from '../hooks/useZoomLevel';

interface ClientLayoutProps {
  children: React.ReactNode;
}

// Check if it's currently night time (7 PM to 6 AM)
const isNightTime = (): boolean => {
  const hour = new Date().getHours();
  return hour >= 19 || hour < 6;
};

export const ClientLayout: React.FC<ClientLayoutProps> = ({ children }) => {
  const [darkMode, setDarkMode] = useState(false);
  const [backgroundMode, setBackgroundMode] = useState<BackgroundMode>('classic');
  const [mounted, setMounted] = useState(false);

  // Initialize zoom detection globally
  useZoomLevel();

  useEffect(() => {
    setMounted(true);
    const savedTheme = localStorage.getItem('theme');
    if (savedTheme) {
      setDarkMode(savedTheme === 'dark');
    } else {
      setDarkMode(isNightTime());
    }
  }, []);

  useEffect(() => {
    if (!mounted) return;

    if (darkMode) {
      document.documentElement.classList.add('dark');
      localStorage.setItem('theme', 'dark');
    } else {
      document.documentElement.classList.remove('dark');
      localStorage.setItem('theme', 'light');
    }
  }, [darkMode, mounted]);

  const toggleTheme = () => setDarkMode(!darkMode);

  return (
    <BackgroundProvider darkMode={darkMode} backgroundMode={backgroundMode}>
      <SmoothScroll>
        <div className="min-h-screen flex flex-col relative selection:bg-brand-orange/20 selection:text-brand-orange">
          {/* Global 3D Background */}
          <div className="fixed inset-0 -z-10">
            {/* Render backgrounds only after mount to avoid hydration issues with 3D components */}
            {mounted && backgroundMode === 'classic' && <GradientBackground />}
            {mounted && backgroundMode === 'blackMagic' && <GradientBackgroundBlackMagic />}
            {mounted && backgroundMode === 'living' && <LivingBackground />}
            {/* Fallback background for SSR */}
            {!mounted && <div className="w-full h-full bg-[#F8FAFC] dark:bg-[#0B1220]" />}
          </div>

          <Header
            darkMode={darkMode}
            toggleTheme={toggleTheme}
            backgroundMode={backgroundMode}
            setBackgroundMode={setBackgroundMode}
          />

          <main className="flex-grow relative z-0">
            {children}
          </main>

          <SecondBrainBanner />
          <Footer />
        </div>
      </SmoothScroll>
    </BackgroundProvider>
  );
};
















