"use client";

import React, { useRef } from "react";
import { Play } from "lucide-react";
import Link from "next/link";
import Image from "next/image";
import { motion, useMotionValue, useSpring, useTransform } from "framer-motion";

type Chapter = {
  chapter_no: number;
  title: string;
  chapter_bg_image: string;
};

type CardProps = {
  chapterNumber: number;
  title: string;
  chapter_bg_image: string;
};

const chapters: Chapter[] = [
  {
    chapter_no: 1,
    title: "Building Trust to Grow a Small to Medium-Sized Organization",
    chapter_bg_image: "/assets/images/chapter_one.jpg",
  },
  {
    chapter_no: 2,
    title: "Leveraging Data to Drive Business Growth",
    chapter_bg_image: "/assets/images/chapter_two.jpg",
  },
  {
    chapter_no: 3,
    title: "Effective Communication for Scaling Teams",
    chapter_bg_image: "/assets/images/chapter_three.jpg",
  },
  {
    chapter_no: 4,
    title: "Customer-Centric Strategies for Growth",
    chapter_bg_image: "/assets/images/chapter_four.jpg",
  },
  {
    chapter_no: 5,
    title: "Leadership and Vision for Expanding Organizations",
    chapter_bg_image: "/assets/images/chapter_five.jpg",
  },
  {
    chapter_no: 6,
    title: "Harnessing Innovation to Stay Competitive",
    chapter_bg_image: "/assets/images/chapter_six.jpg",
  },
  {
    chapter_no: 7,
    title: "Optimizing Processes for Efficiency",
    chapter_bg_image: "/assets/images/chapter_seven.jpg",
  },
  {
    chapter_no: 8,
    title: "Creating a Sustainable Business Model",
    chapter_bg_image: "/assets/images/chapter_eight.jpg",
  },
  {
    chapter_no: 9,
    title: "Empowering Teams for Long-Term Success",
    chapter_bg_image: "/assets/images/chapter_nine.jpg",
  },
];

function ChapterGridForUsecasesBase() {
  return (
    <div className="grid grid-flow-row-dense xl:grid-cols-3 grid-cols-1 md:grid-cols-2 lg:grid-cols-3 w-11/12 mx-auto gap-10">
      {chapters.map((chapter, index) => (
        <ChapterCard
          key={index}
          chapterNumber={chapter.chapter_no}
          title={chapter.title}
          chapter_bg_image={chapter.chapter_bg_image}
        />
      ))}
    </div>
  );
}

function ChapterCard({ chapterNumber, title, chapter_bg_image }: CardProps) {
  const cardRef = useRef<HTMLDivElement>(null);
  
  const x = useMotionValue(0);
  const y = useMotionValue(0);

  // Smooth out the mouse values
  const mouseX = useSpring(x, { stiffness: 300, damping: 30 });
  const mouseY = useSpring(y, { stiffness: 300, damping: 30 });

  const rotateX = useTransform(mouseY, [-0.5, 0.5], [10, -10]);
  const rotateY = useTransform(mouseX, [-0.5, 0.5], [-10, 10]);

  const handleMouseMove = (e: React.MouseEvent<HTMLDivElement>) => {
    if (!cardRef.current) return;

    const card = cardRef.current;
    const rect = card.getBoundingClientRect();
    
    const width = rect.width;
    const height = rect.height;
    
    const mouseXRel = e.clientX - rect.left;
    const mouseYRel = e.clientY - rect.top;

    const xPct = (mouseXRel / width) - 0.5;
    const yPct = (mouseYRel / height) - 0.5;

    x.set(xPct);
    y.set(yPct);
  };

  const handleMouseLeave = () => {
    x.set(0);
    y.set(0);
  };

  return (
    <Link href={`/mindset-grow-your-sme/${chapterNumber}`} className="block h-full">
      <motion.div
        ref={cardRef}
        onMouseMove={handleMouseMove}
        onMouseLeave={handleMouseLeave}
        style={{
          rotateX,
          rotateY,
          transformStyle: "preserve-3d",
          perspective: 1000,
        }}
        whileHover={{ scale: 1.02 }}
        transition={{
          type: "spring",
          stiffness: 300,
          damping: 20,
        }}
        className="relative w-full h-full min-h-[320px] rounded-[2rem] bg-brand-navy border border-white/10 shadow-2xl overflow-hidden group"
      >
        {/* Background Image */}
        <div 
          className="absolute inset-0 w-full h-full transition-transform duration-700 group-hover:scale-110"
          style={{ transform: "translateZ(0px)" }}
        >
          <Image
            src={chapter_bg_image}
            alt={title}
            fill
            className="object-cover opacity-60 group-hover:opacity-40 transition-opacity duration-500"
          />
        </div>

        {/* Gradient Overlay */}
        <div 
          className="absolute inset-0 bg-gradient-to-t from-brand-navy via-brand-navy/80 to-transparent opacity-90"
          style={{ transform: "translateZ(10px)" }}
        />

        {/* Content Container */}
        <div className="relative h-full flex flex-col justify-between p-8 z-10">
          
          {/* Top Section: Chapter Badge */}
          <div 
            className="flex justify-between items-start"
            style={{ transform: "translateZ(30px)" }}
          >
             <span className="inline-flex items-center justify-center px-4 py-1.5 rounded-full bg-brand-blue/20 border border-brand-blue/30 text-brand-blue font-semibold text-xs tracking-widest uppercase backdrop-blur-md shadow-glow">
              Chapter {chapterNumber.toString().padStart(2, '0')}
            </span>

            {/* Play Button */}
            <div 
              className="w-12 h-12 rounded-full bg-white/10 backdrop-blur-md border border-white/20 flex items-center justify-center group-hover:bg-brand-blue group-hover:border-brand-blue transition-all duration-300 shadow-glass"
            >
              <Play className="w-5 h-5 text-white fill-current translate-x-0.5" />
            </div>
          </div>

          {/* Bottom Section: Title */}
          <div 
            className="mt-auto"
            style={{ transform: "translateZ(50px)" }}
          >
            <h3 className="text-2xl md:text-3xl font-bold text-white leading-tight group-hover:text-brand-lightBlue transition-colors duration-300">
              {title}
            </h3>
            
            <div className="h-1 w-12 bg-brand-blue mt-6 rounded-full group-hover:w-24 transition-all duration-500" />
          </div>
        </div>

        {/* Hover Shine Effect */}
        <div
          className="absolute inset-0 pointer-events-none opacity-0 group-hover:opacity-100 transition-opacity duration-500"
          style={{
            background: "radial-gradient(circle at center, rgba(51, 117, 214, 0.15) 0%, transparent 70%)",
            transform: "translateZ(20px)",
          }}
        />
        
      </motion.div>
    </Link>
  );
}

export const ChapterGridForUsecases = Object.assign(ChapterGridForUsecasesBase, {
  Card: ChapterCard,
});


