'use client';

import * as React from 'react';
import Image from 'next/image';

type GradientName = 'purple' | 'blue' | 'orange';
type CardSize = 'sm' | 'md' | 'lg';

type GradientCardProps = React.HTMLAttributes<HTMLDivElement> & {
  gradient?: GradientName;
  size?: CardSize;
};

type GradientCardComponent = React.FC<GradientCardProps> & {
  Title: React.FC<SubProps<'div'>>;
  SubTitle: React.FC<SubProps<'div'>>;
  Content: React.FC<SubProps<'div'>>;
  Action: React.FC<SubProps<'div'>>;
  Media: React.FC<{
    src: string;
    alt?: string;
    className?: string;
    style?: React.CSSProperties;
    priority?: boolean;
  }>;
};

function cx(...parts: Array<string | undefined | false>) {
  return parts.filter(Boolean).join(' ');
}

const gradientBg: Record<GradientName, string> = {
  purple: 'bg-gradient-to-br from-fuchsia-600/80 via-violet-700/60 to-indigo-900/70',
  blue: 'bg-gradient-to-br from-sky-600/70 via-blue-700/60 to-indigo-900/70',
  orange: 'bg-gradient-to-br from-orange-500/70 via-rose-600/50 to-purple-900/60',
};

const sizePadding: Record<CardSize, string> = {
  sm: 'p-0',
  md: 'p-0',
  lg: 'p-0',
};

function GradientCard({
  gradient = 'purple',
  size = 'md',
  className = '',
  children,
  ...props
}: GradientCardProps) {
  return (
    <div className={cx('relative', gradientBg[gradient], sizePadding[size], className)} {...props}>
      {/* Subtle vignette for contrast */}
      <div className="absolute inset-0 bg-black/20" />
      <div className="relative z-10 h-full w-full">{children}</div>
    </div>
  );
}

type SubProps<T extends keyof JSX.IntrinsicElements> = JSX.IntrinsicElements[T] & {
  className?: string;
};

const GradientCardTyped = GradientCard as unknown as GradientCardComponent;

GradientCardTyped.Title = function Title({ className = '', ...props }: SubProps<'div'>) {
  return <div className={cx('text-sm font-semibold uppercase tracking-wider text-white/90', className)} {...props} />;
};

GradientCardTyped.SubTitle = function SubTitle({ className = '', ...props }: SubProps<'div'>) {
  return <div className={cx('text-white', className)} {...props} />;
};

GradientCardTyped.Content = function Content({ className = '', ...props }: SubProps<'div'>) {
  return <div className={cx('relative', className)} {...props} />;
};

GradientCardTyped.Action = function Action({ className = '', ...props }: SubProps<'div'>) {
  return <div className={cx('absolute bottom-8 right-8', className)} {...props} />;
};

GradientCardTyped.Media = function Media({
  className = '',
  src,
  alt = '',
  style,
  priority,
}: {
  src: string;
  alt?: string;
  className?: string;
  style?: React.CSSProperties;
  priority?: boolean;
}) {
  return (
    <Image
      src={src}
      alt={alt}
      fill
      priority={priority}
      sizes="(max-width: 768px) 100vw, (max-width: 1280px) 50vw, 33vw"
      className={cx('object-cover', className)}
      style={style}
    />
  );
};

export default GradientCardTyped;


