'use client';

import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Mail, MessageSquare, Globe, Phone, Zap, Wifi, ArrowDown, Shield, Radio } from 'lucide-react';
import { CardShell } from '../shared/CardShell';

// --- Mock Data ---
const SOURCES = [
    { id: 'web', icon: Globe, color: 'text-indigo-500', bg: 'bg-indigo-500', label: 'Website' },
    { id: 'chat', icon: MessageSquare, color: 'text-emerald-500', bg: 'bg-emerald-500', label: 'Live Chat' },
    { id: 'email', icon: Mail, color: 'text-blue-500', bg: 'bg-blue-500', label: 'Email' },
    { id: 'voip', icon: Phone, color: 'text-violet-500', bg: 'bg-violet-500', label: 'VoIP' },
];

const INQUIRIES = [
    { text: "Quote for 500 units of...", source: 0 },
    { text: "Urgent support needed for...", source: 1 },
    { text: "Partnership opportunity...", source: 2 },
    { text: "Callback request...", source: 3 },
];

export const CaptureVisual = () => {
    const [activeSource, setActiveSource] = useState<number | null>(null);
    const [items, setItems] = useState<any[]>([]);
    const [scanLine, setScanLine] = useState(0);

    // Simulation Loop
    useEffect(() => {
        const interval = setInterval(() => {
            // Randomly pick a source to "fire"
            const sourceIdx = Math.floor(Math.random() * SOURCES.length);
            setActiveSource(sourceIdx);

            // Add item to feed
            const newItem = {
                id: Date.now(),
                source: SOURCES[sourceIdx],
                
                text: INQUIRIES[Math.floor(Math.random() * INQUIRIES.length)].text,
                time: new Date().toLocaleTimeString([], { hour12: false, hour: '2-digit', minute: '2-digit', second: '2-digit' }),
                status: 'analyzing'
            };

            setItems(prev => [newItem, ...prev].slice(0, 5));

            // Reset source highlight
            setTimeout(() => setActiveSource(null), 800);

        }, 2000);

        return () => clearInterval(interval);
    }, []);

    return (
        <CardShell>
            <div className="h-[600px] relative bg-slate-50 dark:bg-slate-900 overflow-hidden flex flex-col font-sans">

                {/* --- Background Effects --- */}
                <div className="absolute inset-0 opacity-[0.03]"
                    style={{ backgroundImage: 'radial-gradient(#64748b 1px, transparent 1px)', backgroundSize: '24px 24px' }} />

                {/* Radar Sweep Effect */}
                <div className="absolute inset-0 overflow-hidden pointer-events-none">
                    <motion.div
                        animate={{ rotate: 360 }}
                        transition={{ duration: 8, repeat: Infinity, ease: "linear" }}
                        className="absolute -top-[50%] -left-[50%] w-[200%] h-[200%] bg-gradient-to-r from-transparent via-blue-500/5 to-transparent"
                        style={{ transformOrigin: '50% 50%' }}
                    />
                </div>

                {/* --- Header --- */}
                <div className="relative z-10 flex justify-between items-center p-5 border-b border-slate-200/50 dark:border-slate-700/50 backdrop-blur-sm bg-white/50 dark:bg-slate-900/50">
                    <div className="flex items-center gap-3">
                        <div className="flex items-center justify-center w-8 h-8 rounded-lg bg-white dark:bg-slate-800 shadow-sm border border-slate-200 dark:border-slate-700">
                            <Zap className="w-4 h-4 text-amber-500 fill-amber-500" />
                        </div>
                        <div>
                            
                            <div className="text-xs font-bold text-slate-800 dark:text-slate-100 flex items-center gap-1.5 uppercase tracking-wider">    <span className="relative flex h-2 w-2">
                                    <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-emerald-400 opacity-75"></span>
                                    <span className="relative inline-flex rounded-full h-2 w-2 bg-emerald-500"></span>
                                </span>Live Ingest</div>
                            {/* <div className="flex items-center gap-1.5">
                             
                                <span className="text-[10px] font-mono text-slate-500 dark:text-slate-400">WS_SECURE_V4</span>
                            </div> */}
                        </div>
                    </div>
                    <div className="flex items-center gap-2 px-3 py-1.5 rounded-full bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700">
                        <Wifi className="w-3 h-3 text-green-500" />
                        <span className="text-[10px] font-mono font-bold text-slate-600 dark:text-slate-300">ONLINE</span>
                    </div>
                </div>

                {/* --- Source Nodes (Top) --- */}
                <div className="flex justify-center gap-6 py-8 relative z-10">
                    {SOURCES.map((s, i) => {
                        const isActive = activeSource === i;
                        return (
                            <div key={s.id} className="relative flex flex-col items-center gap-3 ">
                                {/* Connection Beam */}
                                <motion.div
                                    initial={{ height: 0, opacity: 0 }}
                                    animate={{ height: isActive ? 90 : 0, opacity: isActive ? 1 : 0 }}
                                    className={`absolute top-12 w-0.5 ${s.bg} blur-[1px]`}
                                />

                                {/* Node Icon */}
                                <motion.div
                                    animate={isActive ? { scale: [1, 1.1, 1], boxShadow: "0 0 20px rgba(59, 130, 246, 0.5)" } : {}}
                                    className={`
                                        w-14 h-14 rounded-2xl flex items-center justify-center
                                        bg-white dark:bg-slate-800 border-2 transition-colors duration-300
                                        ${isActive ? `border-${s.color.split('-')[1]}-500` : 'border-slate-200 dark:border-slate-700'}
                                        shadow-xl z-20
                                    `}
                                >
                                    <s.icon className={`w-6 h-6 ${s.color}`} />
                                </motion.div>

                                <span className="text-[10px] font-bold uppercase text-slate-400 tracking-wider">{s.label}</span>

                                {/* Particle Drop */}
                                <AnimatePresence>
                                    {isActive && (
                                        <motion.div
                                            initial={{ y: 0, scale: 0 }}
                                            animate={{ y: 100, scale: 1 }}
                                            exit={{ opacity: 0 }}
                                            transition={{ duration: 0.6, ease: "circIn" }}
                                            className={`absolute top-14 w-3 h-3 rounded-full ${s.bg} shadow-lg z-10`}
                                        />
                                    )}
                                </AnimatePresence>
                            </div>
                        )
                    })}
                </div>

                {/* --- Ingest Feed (Bottom) --- */}
                <div className="flex-1 px-6 pb-6 relative z-10 overflow-hidden flex flex-col justify-end">
                    <div className="absolute top-0 left-0 w-full h-12 bg-gradient-to-b from-slate-50 dark:from-slate-900 to-transparent z-20 pointer-events-none" />

                    <div className="flex items-center gap-2 mb-2 px-2 opacity-50">
                        <ArrowDown size={12} className="text-slate-400 animate-bounce" />
                        <span className="text-[10px] font-bold uppercase text-slate-400">Live Stream</span>
                    </div>

                    <div className="space-y-3 min-h-[300px] flex flex-col justify-end">
                        <AnimatePresence mode="popLayout">
                            {items.map((item, i) => (
                                <motion.div
                                    key={item.id}
                                    layout
                                    initial={{ opacity: 0, y: -50, scale: 0.9, filter: "blur(10px)" }}
                                    animate={{ opacity: 1, y: 0, scale: 1, filter: "blur(0px)" }}
                                    exit={{ opacity: 0, scale: 0.95 }}
                                    className={`
                                        relative overflow-hidden pb-8 p-2 rounded-xl border backdrop-blur-md
                                        bg-white/80 dark:bg-slate-800/80
                                        border-slate-200 dark:border-slate-700
                                        shadow-sm group
                                    `}
                                >
                                    {/* Scan Sweep on Entry */}
                                    <motion.div
                                        initial={{ x: "-100%" }}
                                        animate={{ x: "200%" }}
                                        transition={{ duration: 1.5, ease: "linear" }}
                                        className="absolute inset-0 bg-gradient-to-r from-transparent via-white/40 dark:via-white/10 to-transparent skew-x-12 pointer-events-none"
                                    />

                                    <div className="flex items-center gap-4 relative z-10">
                                        <div className={`p-2 rounded-lg ${item.source.bg} bg-opacity-10 dark:bg-opacity-20`}>
                                            <item.source.icon size={16} className={item.source.color} />
                                        </div>
                                        <div className="flex-1">
                                            <div className="flex justify-between items-start">
                                                <h4 className="text-sm font-bold text-slate-800 dark:text-white">
                                                    {item.source.id === 'web'
                                                        ? 'Inquiry Captured'
                                                        : ['chat', 'email', 'voip'].includes(item.source.id)
                                                            ? 'New Inquiry Added'
                                                            : 'New Inquiry Detected'}
                                                </h4>
                                                <span className="text-[10px] font-mono text-slate-400">{item.time}</span>
                                            </div>
                                            <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">{item.text}</p>
                                        </div>

                                        {/* Status Tag */}
                                        <div className="flex flex-col items-end gap-1">
                                            <div className="px-2 py-0.5 rounded w-20 text-[10px] font-bold bg-blue-50 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 border border-blue-100 dark:border-blue-800 flex items-center gap-1">
                                                <Zap size={10} />{' '}
                                                {item.source.id === 'web'
                                                    ? 'CAPTURED'
                                                    : ['chat', 'email', 'voip'].includes(item.source.id)
                                                        ? 'ADDED'
                                                        : 'CAPTURED'}
                                            </div>
                                        </div>
                                    </div>
                                </motion.div>
                            ))}
                        </AnimatePresence>
                    </div>
                </div>

            </div>
        </CardShell>
    );
};
