export interface ColorPalette {
  colorA: string;
  colorB: string;
  colorC: string;
  colorD: string;
  background: string;
}

export type ThemeId = 'hero' | 'solutions' | 'mindset' | 'pricing' | 'contact' | 'default';

export interface ThemeConfig {
  light: ColorPalette;
  dark: ColorPalette;
}

// Background "style" modes (global background variants toggled via the logo easter-egg).
export type BackgroundMode = 'classic' | 'blackMagic' | 'living';

// Central QERP brand colors (mirrors Tailwind config in index.html)
const BRAND = {
  blue: '#0052CC',
  lightBlue: '#3375D6',
  orange: '#FE6C00',
  lightOrange: '#F19058',
  teal: '#00CEC2',
  lightTeal: '#7BD3D0',
  navy: '#0B1220',
} as const;

// Palettes drive the 3D gradient background + section theming.
// Each theme keeps the base neutral SaaS feel but leans into QERP brand blues/teals/oranges.
export const PALETTES: Record<ThemeId, ThemeConfig> = {
  default: {
    light: {
      // Calm SaaS shell used on most pages
      colorA: '#E2E8F0', // Slate 200
      colorB: '#F1F5F9', // Slate 100
      colorC: '#FFFFFF', // Card surfaces
      colorD: BRAND.blue, // Very soft brand accent in the mesh
      background: '#F8FAFC', // App background
    },
    dark: {
      colorA: '#020617', // Slate 950
      colorB: BRAND.navy, // Brand base
      colorC: '#1E293B', // Slate 800
      colorD: BRAND.lightBlue, // Accent
      background: '#020617',
    },
  },
  hero: {
    light: {
      // Hero: QERP headline mesh – cool, optimistic, product-first
      colorA: '#DBEAFE', // Blue 100
      colorB: '#E0F2FE', // Sky 100
      colorC: '#CCFBF1', // Teal 100
      colorD: BRAND.blue, // Stronger brand core under the logo/dashboard
      background: '#FFFFFF',
    },
    dark: {
      // Hero dark: deep navy with brand blue + teal glow
      colorA: BRAND.navy,
      colorB: BRAND.blue,
      colorC: BRAND.lightBlue,
      colorD: BRAND.teal,
      background: '#020617',
    },
  },
  solutions: {
    light: {
      // Modules & flows: analytics / execution feel using teal + sky
      colorA: '#CCFBF1', // Teal 100
      colorB: '#E0F2FE', // Sky 100
      colorC: '#EFF6FF', // Blue 50
      colorD: BRAND.teal, // QERP execution accent
      background: '#FFFFFF',
    },
    dark: {
      // Dark analytics canvas
      colorA: '#022C22', // Deep teal
      colorB: BRAND.navy,
      colorC: '#0F766E', // Teal 700
      colorD: BRAND.lightTeal,
      background: '#020617',
    },
  },
  mindset: {
    light: {
      // Content-heavy, slightly warmer but still product-adjacent
      colorA: '#F9FAFB', // Neutral 50
      colorB: '#E5E7EB', // Neutral 200
      colorC: '#FEF3C7', // Amber 100 – subtle warmth
      colorD: BRAND.orange, // Growth / energy accent
      background: '#FFFFFF',
    },
    dark: {
      colorA: '#111827', // Gray 900
      colorB: '#020617',
      colorC: '#1F2937', // Gray 800
      colorD: BRAND.lightOrange,
      background: '#020617',
    },
  },
  pricing: {
    light: {
      // Pricing: premium but clearly on-brand
      colorA: '#EEF2FF', // Indigo 50
      colorB: '#E0E7FF', // Indigo 100
      colorC: '#F8FAFC', // Neutral wash
      colorD: BRAND.blue, // Emphasise brand trust
      background: '#FAFAFA',
    },
    dark: {
      colorA: '#020617',
      colorB: BRAND.navy,
      colorC: '#1E293B',
      colorD: BRAND.orange, // Call-to-action glow
      background: '#020617',
    },
  },
  contact: {
    light: {
      // Contact: clean and minimal, with a blue accent
      colorA: '#F8FAFC',
      colorB: '#FFFFFF',
      colorC: '#E2E8F0',
      colorD: BRAND.blue,
      background: '#FFFFFF',
    },
    dark: {
      colorA: '#020617',
      colorB: BRAND.navy,
      colorC: '#1E293B',
      colorD: BRAND.teal,
      background: '#020617',
    },
  },
};

/**
 * Background-mode palettes.
 *
 * These are intentionally different from section palettes:
 * - Section palettes (`PALETTES`) change as you scroll (via `useSectionTheme`)
 * - Background palettes (`BACKGROUND_MODE_PALETTES`) change when switching the global background mode
 *
 * BlackMagic + Living backgrounds are shader-driven and read these values directly.
 */
export const BACKGROUND_MODE_PALETTES: Record<BackgroundMode, ThemeConfig> = {
  classic: {
    light: {
      // Airy + friendly (matches the soft blob gradient vibe)
      colorA: '#DBEAFE', // Blue 100
      colorB: '#E0E7FF', // Indigo 100
      colorC: '#FCE7F3', // Pink 100
      colorD: '#FEF3C7', // Amber 100
      background: '#F8FAFC', // Slate 50
    },
    dark: {
      // Deep but calm; blue glow keeps it product-y
      colorA: '#020617', // Slate 950
      colorB: BRAND.navy,
      colorC: '#1E293B', // Slate 800
      colorD: BRAND.lightBlue,
      background: '#020617',
    },
  },
  blackMagic: {
    light: {
      // Crisp "glass" canvas with a slightly more electric spectrum
      colorA: '#EEF2FF', // Indigo 50
      colorB: '#DBEAFE', // Blue 100
      colorC: '#E9D5FF', // Purple 200 for a bit more depth behind text
      colorD: '#99F6E4', // Teal 200 to add contrast/punch
      background: '#FFFFFF',
    },
    dark: {
      // Dramatic, high-contrast, cyber-ish. Teal/blue peaks feel premium.
      colorA: BRAND.navy,
      colorB: BRAND.blue,
      colorC: '#312E81', // Indigo 900
      colorD: BRAND.teal,
      background: '#020617',
    },
  },
  living: {
    light: {
      // Organic aurora; cooler + clean to keep content readable
      colorA: '#E0F2FE', // Sky 100
      colorB: '#CCFBF1', // Teal 100
      colorC: '#F0ABFC', // Fuchsia 300 for safer text contrast
      colorD: '#DBEAFE', // Blue 100
      background: '#F7FAFD', // Slight off-white to soften glare
    },
    dark: {
      // Organic but grounded; teal/sky peaks with deep slate base
      colorA: '#022C22', // Deep teal
      colorB: BRAND.navy,
      colorC: '#0F172A', // Slate 900
      colorD: BRAND.lightTeal,
      background: '#020617',
    },
  },
};
